% object that loads and administers all of the words and their defs.
%
% copyright 2011-2012 Blair Armstrong, Natasha Tokowicz, David Plaut
%
%    This file is part of eDom
%
%    eDom is free software: you can redistribute it and/or modify
%    it for academic and non-commercial purposes
%    under the terms of the GNU General Public License as published by
%    the Free Software Foundation, either version 3 of the License, or
%    (at your option) any later version.  For commercial or for-profit
%    uses, please contact the authors (blairarm@andrew.cmu.edu).
%
%    eDom is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%    GNU General Public License for more details.

%    You should have received a copy of the GNU General Public License
%    along with eDom (see COPYING.txt).
%    If not, see <http://www.gnu.org/licenses/>.

classdef defAdmin < handle
    
    properties
        defs        
    end
    
    methods
        
        %constructor
        function obj = defAdmin(wordFile,defDir,order,noDefsAction)
 
            defs = {};  %#ok<PROP,NASGU>

            % get the list of words for this participant
            wordfid = fopen(wordFile);
            wordList = textscan(wordfid,'%s');
            wordList = wordList{1};
            
            % randomize the ordering of that list if appropriate; 
            if strcmp(order,'random')
                perm = randperm(length(wordList));
                wordList = wordList(perm);
            elseif strcmp(order,'fixed')
                % do nothing
            else
                error('Invalid "order"');
            end

            % get a list of all the definition files
            % from the folder containing definitions
            dirList = dir(defDir);
            
            %extract file names from dir struct array
            dirNames = cell(length(dirList),1);
            for i=1:length(dirList)
                dirNames{i} = dirList(i).name;
            end            
            
            % get a lowercase copy of the names so that we can compare the words to the
            % lowercase version, but also refer to the correct case when attempting to
            % access the different folders.
            dirNamesLowerParsed = lower(dirNames);

            for i=1:length(dirNamesLowerParsed)

               tmp = regexp(dirNamesLowerParsed{i},'^[^.]+','match');
               dirNamesLowerParsed{i} = char(tmp);
            end



            % retrieve the definitions for each word from their files
            nMeaningList = [];

            for i=1:length(wordList)
                disp(['Processing item: ',wordList{i},' (',num2str(i), ...
                        ' of ', num2str(length(wordList)),')']);

                targMatches = strcmp(dirNamesLowerParsed,wordList{i});        
                targIndicies = find(targMatches);

                % fail if no defs for a word and user said this was not
                % allowed
                if isempty(targIndicies) && strcmp(noDefsAction,'denyNoDefs')
                    error(['Failed on item: ',wordList{i}]);
                end

                targFiles = (dirNames(targIndicies));


                if isempty(targFiles) && strcmp(noDefsAction,'denyNoDefs')
                        error(['unable to find unique word in dictionary files: ',wordList{i}]);
                end
                
                %remove special files that end in .all.txt (from wordsmyth
                % parsing)
                for z=1:length(targFiles)
                   if isempty(regexp(targFiles{z},['^.*.all.txt$'],'match')) ~= 1 %#ok<NBRAK>
                       targFiles(z) = [];
                   end
                end

                totalFiles = length(targFiles); 
                cFiles = 0; % how many files have been counted so far.  

                mEntries = {};
                totalMeanings = 0;
                n = 1;       

                % retrieve a list of the files denoting different senses of a
                % particular meaning for each word and store these lists of
                % files seperately.  Also make sure that we've accounted for
                % each of the meanings in sequence.  
                while cFiles < totalFiles
                    mEntries{n} = regexp(targFiles,['^.+\.',num2str(n),'\.((.+[^l])|())\.txt$'],'match'); %#ok<AGROW>
                    
                    minOne = false;
                    for k=1:length(mEntries{n})
                        if ~isempty(mEntries{n}{k})
                            minOne = true;
                            cFiles = cFiles + 1;
                        end
                    end

                    % make sure at least one entry was found for each meaning in
                    % the sequence 1...n meanings
                    if minOne == false
                        error(['Missing a meaning in sequence 1..n for: ',wordList{i}]);
                    else
                        totalMeanings = totalMeanings + 1;
                    end

                    n=n+1;
                end    
                
                nMeaningList  = [nMeaningList; totalMeanings]; %#ok<AGROW>
                
                % definitions of each interpretation.
                intDefs = {};
                
                for k=1:length(mEntries)
                    % read all the files that contain definitions and prepare a
                    % list containing all of those definitions.  
                    intDefs{k} = getAllDefinitions([defDir,'/'],mEntries{k}); %#ok<AGROW>

                end
                
                % we now have the definitions for the word stored in
                % intDefs for each definition, separately.  Store these in
                % separate objects that can be iterated through...
                
                def = wordDef(wordList{i},intDefs);
                
                obj.defs = [obj.defs; def];
             
            end % iterating through each words
       
        end % end defAdmin constructor
        
        
 
        % can pop - checks if there are still more items in the defAdmin
        % object
        function flag = canPop(obj)
            if isempty(obj.defs)
                flag = 0;
            else
                flag = 1;
            end
        end
            
        % returns the next object in the stack and shortens the list by 1
        function defn = pop(obj)
            defn = obj.defs(1);
            
            obj.defs = obj.defs(2:length(obj.defs));
        end
            
    end
    
end
